function [ut,auxt,auxt_labels]=regulator_feedforward_xest(i,t,x,y,u,ts,csi,o,aux)

% Feedforward control strategy using biomass concentration estimates by software sensors 
%
% [ut,auxt,auxt_labels]=regulator_feedforward_xest(i,t,x,y,u,ts,csi,o,aux)
%
% OUTPUT
% ut ... row vector of values of manipulated variables at time t
% auxt ... row vector of auxiliary variables for optional storage of time-varying variables and parameters of control strategies (e.g. setpoints)
%
% ut(1) ... feed flowrate (F, ml/s)
% ut(2) ... air flowrate (Fv, ml/s)  
% ut(3) ... stirrer speed (n, 1/s)
%
% auxt(1) ... auxiliary variable no. 1
% auxt(2) ... auxiliary variable no. 2
% auxt(3) ... auxiliary variable no. 3
% auxt(4) ... auxiliary variable no. 4
% auxt(5) ... auxiliary variable no. 5
%
% auxt_labels(1) ... text label for auxiliary variable no. 1
% auxt_labels(2) ... text label for auxiliary variable no. 2
% auxt_labels(3) ... text label for auxiliary variable no. 3
% auxt_labels(4) ... text label for auxiliary variable no. 4
% auxt_labels(5) ... text label for auxiliary variable no. 5
%
% INPUT
% i ... integer index value passed from the parent FOR loop
% t ... current time (s)
% x ... matrix of state variables
% y ... matrix of output (measured) variables
% u ... matrix of manipulated variables
% ts ... sample time (s)
% csi ... glucose concentration in the feed (mol/ml)
% o ... matrix of estimated variables by soft sensors
% aux ... matrix of auxiliary variables for optional storage of time-varying variables and parameters of control strategies (e.g. setpoints)
% Note: for the composition of the x,y,u,o,aux matrices, see help for yeast_simcon.m function

global citac_spusteni_reg

ut=zeros(1,3);
auxt=zeros(1,5)*NaN;
auxt_labels={'auxiliary variable no. 1','auxiliary variable no. 2','auxiliary variable no. 3','auxiliary variable no. 4','auxiliary variable no. 5'};


% air flowrate set to a constant value of 10 l/min, i.e. 167 ml/s
ut(2)=167;

% stirrer speed set to a constant value of 600 1/min, i.e. 10 1/s
ut(3)=10;

% desired biomass growth rate (1/s)
kw=0.25/3600;

% desired biomass yield per substrate (mol X/mol S)
Yxsw=0.47;

% choice of softsensor type for biomass concentration estimation
% 1 ... estimation of biomass concentration value (mol/ml) from consumption and production rates (O2, CO2, substrate)
% 2 ... estimation of biomass concentration value (mol/ml) from cumulative O2 consumption

volba_softsens=1;
odhad_konc_biomasy=o(i,volba_softsens);

% feed flowrate calculation (ml/s)
ut(1)=kw*odhad_konc_biomasy*x(i,6)/Yxsw/csi;
citac_spusteni_reg=citac_spusteni_reg+1;

% safety check against overfilling of the bioreactor

% maximum total allowable volume of liquid in the bioreactor (ml)
vol_max=7500;

cur_vol=x(i,6);
new_vol=cur_vol+ut(1)*60;

if new_vol>vol_max
	ut(1)=(vol_max-cur_vol)/60;
end
