function [y,x,u,o,aux,aux_labels]=yeast_simcon(t0,t1,ts,x0,csi,reg)

% Main function of the bioprocess simulator (modeled process: fed-batch cultivation of the yeast Saccharomyces cerevisae, process model is based on the well-known theory of limited respiratory capacity developed by (Sonnleitner and Kppeli, 1986) and (Sonnleitner and Hahnemann, 1994))
%
% [y,x,u,o,aux,aux_labels]=yeast_simcon(t0,t1,ts,x0,csi,reg)
%
% OUTPUTS
% y ... matrix of values of output (measured) variables for time interval <t0,t1> with sampling time ts
% x ... matrix of values of state variables for time interval <t0,t1> with sampling time ts
% u ... matrix of values of manipulated variables for time interval <t0,t1> with sampling time ts
% o ... matrix of values of variables estimated by soft sensors for time interval <t0,t1> with sampling time ts
% aux ... matrix of auxiliary variables for optional storage of time-varying variables and parameters of control strategies (e.g. setpoints) for time interval <t0,t1> with sampling time ts
% aux_labels ... text labels of auxiliary variables
%
% y(:,1) ... ethanol mol fraction in the fermentation broth (xe, %)
% y(:,2) ... O2 mol fraction in the bioreactor off-gas (xoog, %)
% y(:,3) ... CO2 mol fraction in the bioreactor off-gas (xcog, %)
% y(:,4) ... dissolved O2 concentration in the fermentation broth (DO, % saturation)
%
% x(:,1) ... biomass concentration in the fermentation broth (cx, mol/ml)
% x(:,2) ... glucose concentration in the fermentation broth (cs, mol/ml)
% x(:,3) ... ethanol concentration in the fermentation broth (ce, mol/ml)
% x(:,4) ... O2 concentration in the fermentation broth (co, mol/ml)
% x(:,5) ... CO2 concentration in the fermentation broth (cc, mol/ml)
% x(:,6) ... fermentation broth volume (vt, ml)
% x(:,7) ... available capacity of specific oxygen consumption (Sonnleitner and Hahnemann, 1994) (qocap, 1/s)
% x(:,8) ... auxiliary indicator of respiration adaptation to high ethanol concentration (Sonnleitner and Hahnemann, 1994) (help)
% x(:,9) ... auxiliary indicator of respiration suppression due to high ethanol concentration (Sonnleitner and Hahnemann, 1994) (sup)
%
% u(:,1) ... feed flowrate (F, ml/s) (according to a specific feeding strategy)
% u(:,2) ... air flowrate (Fv, ml/s) (in a laboratory fermenter typically 10 l/min, i.e. 167 ml/s)  
% u(:,3) ... stirrer speed (n, 1/s) (in a laboratory fermenter typically 600 1/min, i.e. 10 1/s)
%
% o(:,1) ... estimation of the biomass concentration value (mol/ml) using a soft sensor from consumption and production rates (O2, CO2, substrate)
% o(:,2) ... estimation of biomass concentration value (mol/ml) using a soft sensor from cumulative O2 consumption
%
% aux(:,1) ... auxiliary variable no. 1
% aux(:,2) ... auxiliary variable no. 2
% aux(:,3) ... auxiliary variable no. 3
% aux(:,4) ... auxiliary variable no. 4
% aux(:,5) ... auxiliary variable no. 5
%
% aux_labels(1) ... text label for auxiliary variable no. 1
% aux_labels(2) ... text label for auxiliary variable no. 2
% aux_labels(3) ... text label for auxiliary variable no. 3
% aux_labels(4) ... text label for auxiliary variable no. 4
% aux_labels(5) ... text label for auxiliary variable no. 5
%
% INPUTS
% t0 ... start time of simulated cultivation (s)
% t1 ... end time of simulated cultivation (s)
% ts ... sampling time (s)
%
% x0 ... initial condition at time t0
% typically x0=[cx0 0 0 0.23e-6 0 v0 0.2/3600 0 0]
% where cx0 is the initial biomass concentration (e.g. 1 g/l, i.e. 4e-5 mol/ml)
% and v0 is the initial volume (typically 5 l in a lab fermenter, i.e. 5000 ml)
% for the meaning of other entries of x0, see the state vector x in OUTPUTS
% csi ... concentration of glucose in the nutrient solution (mol/ml) (in a lab fermenter typically 100 g/l, i.e. 0.0033 mol/ml)
% reg ... m-function defining the control strategy (in the form ut=regulator(i,t,x,y,u))
%
% Molar masses of biomass and substrate considered in the model
% biomass M(x)=25.17 g/mol
% glucose M(s)=30 g/mol


nx=9;
ny=4;
nu=3;
no=2;
naux=5;

if length(x0)~=nx
	error('Incorrectly defined initial condition!');
end

n=round((t1-t0)/ts);
xt=x0;
x=zeros(n+1,nx);
y=zeros(n+1,ny);
u=zeros(n+1,nu);
o=zeros(n+1,no);
aux=zeros(n+1,naux);
x(1,:)=xt;

% Setting the parameters for the ODE solver

accur=1e-9;	% required accuracy of the numerical solution
hini=0.1;	% initial step (s)
hmax=ts;	% maximum step (s)
hmin=3.6e-3;	% minimum step (s)

for i=1:n
	t=t0+(i-1)*ts;
	o(i,:)=odhad_softsens(i,t,x,y,u,ts,csi);
	[ut,auxt,aux_labels]=reg(i,t,x,y,u,ts,csi,o,aux);
	u(i,:)=ut;
	aux(i,:)=auxt;
	pt=get_par(xt,ut);
	pt(14)=csi;
	xt=merson(t,t+ts,accur,hini,hmax,hmin,xt,ut,pt);
	for j=1:nx-1
		if xt(j)<0
			xt(j)=0;
		end
	end
	yt=g(xt,ut,pt);

	x(i+1,:)=xt;
	y(i+1,:)=yt;

	if i==1
		y(i,:)=yt;
	end
end

o(i+1,:)=o(i,:);
u(i+1,:)=ut;
aux(i+1,:)=auxt;


